import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object target positions based on retrieved positions
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        bottom_side_pos = positions['bottom_side_pos']  # Position for moving to the side of the drawer
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # === Step 1: Rotate the gripper to the correct orientation ===
        print("[Task] Rotating gripper to proper orientation.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Rotate from zero_deg to ninety_deg

        if done:
            print("[Task] Gripper rotated successfully.")
        else:
            print("[Task] Failed to rotate gripper.")
            return

        # === Step 2: Move to the side position to access the drawer ===
        print("[Task] Moving to side position to access the drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)

        if done:
            print("[Task] Moved to side position successfully.")
        else:
            print("[Task] Failed to move to side position.")
            return

        # === Step 3: Move to anchor position for the drawer ===
        print("[Task] Moving to anchor position for the drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)

        if done:
            print("[Task] Moved to anchor position successfully.")
        else:
            print("[Task] Failed to move to anchor position.")
            return

        # === Step 4: Pick the drawer ===
        print("[Task] Picking the drawer.")
        obs, reward, done = pick(env, task, 'bottom', 'drawer')

        if done:
            print("[Task] Drawer picked successfully.")
        else:
            print("[Task] Failed to pick the drawer.")
            return

        # === Step 5: Pull the drawer to access its contents ===
        print("[Task] Pulling the drawer to access its contents.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')

        if done:
            print("[Task] Drawer pulled successfully.")
        else:
            print("[Task] Failed to pull the drawer.")
            return
        
        # === Step 6: Picking up the rubbish from the table ===
        print("[Task] Picking up rubbish from the table.")
        obs, reward, done = pick(env, task, rubbish_pos, 'table')

        if done:
            print("[Task] Rubbish picked successfully.")
        else:
            print("[Task] Failed to pick rubbish.")
            return
        
        # === Step 7: Placing the rubbish into the bin ===
        print("[Task] Placing rubbish into the bin.")
        obs, reward, done = place(env, task, rubbish_pos, bin_pos)

        if done:
            print("[Task] Rubbish placed into the bin successfully. Goal achieved!")
        else:
            print("[Task] Failed to place rubbish into the bin.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()