import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define positions of objects for clarity
        bottom_drawer_pos = positions['bottom_anchor_pos']
        plate_pos = positions['plate']
        bottom_side_pos = positions['bottom_side_pos']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']

        # === Task Plan Execution ===
        print("[Task] Rotating gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05)
        if done:
            print("[Task] Gripper rotated successfully.")
        else:
            print("[Task] Failed to rotate the gripper before timeout.")
            return

        print("[Task] Moving to side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos, max_steps=100, threshold=0.01)
        if done:
            print("[Task] Moved successfully to the side position.")
        else:
            print("[Task] Failed to move to the side position before timeout.")
            return

        print("[Task] Moving to anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos, max_steps=100, threshold=0.01)
        if done:
            print("[Task] Moved successfully to the anchor position.")
        else:
            print("[Task] Failed to move to the anchor position before timeout.")
            return

        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos, approach_distance=0.15)
        if done:
            print("[Task] Bottom drawer picked successfully.")
        else:
            print("[Task] Failed to pick the bottom drawer.")
            return
        
        print("[Task] Pulling the bottom drawer to unlock.")
        obs, reward, done = pull(env, task, pull_distance=0.15, pull_axis='y', max_steps=100, threshold=0.01)
        if done:
            print("[Task] Drawer successfully pulled.")
        else:
            print("[Task] Failed to pull the drawer before timeout.")
            return

        print("[Task] Picking the first tomato.")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos, approach_distance=0.15)
        if done:
            print("[Task] First tomato picked.")
        else:
            print("[Task] Failed to pick the first tomato.")
            return
        
        print("[Task] Placing the first tomato on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15)
        if done:
            print("[Task] First tomato placed on the plate.")
        else:
            print("[Task] Failed to place the first tomato.")
            return
        
        print("[Task] Picking the second tomato.")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos, approach_distance=0.15)
        if done:
            print("[Task] Second tomato picked.")
        else:
            print("[Task] Failed to pick the second tomato.")
            return
        
        print("[Task] Placing the second tomato on the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15)
        if done:
            print("[Task] Second tomato placed on the plate.")
        else:
            print("[Task] Failed to place the second tomato.")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()