import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, pull, rotate, move

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Extract the relevant positions from the retrieved object positions
        plate_pos = positions['plate']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        bottom_anchor_pos = positions['bottom_anchor_pos']  # Anchor position for the bottom drawer
        bottom_side_pos = positions['bottom_side_pos']      # Side position for the bottom drawer
        
        # Step 1: Rotate to face the bottom drawer (initial position adjustment)
        print("[Task] Rotating gripper to face the bottom drawer.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 0, 1]))  # Assuming quaternion for no rotation
        if done:
            print("[Task] Rotation completed.")
        else:
            print("[Task] Failed to rotate.")
            return

        # Step 2: Move to the side position near the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)  
        if done:
            print("[Task] Moved to side position.")
        else:
            print("[Task] Failed to move to side position.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)  
        if done:
            print("[Task] Moved to anchor position.")
        else:
            print("[Task] Failed to move to anchor position.")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Attempting to pick up the bottom drawer.")
        obs, reward, done = pick(env, task, obj='bottom', loc='bottom_anchor_pos')
        if done:
            print("[Task] Bottom drawer picked successfully!")
        else:
            print("[Task] Failed to pick the bottom drawer.")
            return

        # Step 5: Pull the bottom drawer open
        print("[Task] Pulling the bottom drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')  
        if done:
            print("[Task] Drawer opened successfully!")
        else:
            print("[Task] Failed to open the drawer.")
            return

        # Step 6: Pick the first tomato from the table
        print("[Task] Picking tomato1 from the table.")
        obs, reward, done = pick(env, task, obj='tomato1', loc='table')
        if done:
            print("[Task] tomato1 picked successfully!")
        else:
            print("[Task] Failed to pick tomato1.")
            return

        # Step 7: Place tomato1 on the plate
        print("[Task] Placing tomato1 on the plate.")
        obs, reward, done = place(env, task, obj='tomato1', loc='plate')
        if done:
            print("[Task] tomato1 placed successfully!")
        else:
            print("[Task] Failed to place tomato1.")
            return

        # Step 8: Pick the second tomato from the table
        print("[Task] Picking tomato2 from the table.")
        obs, reward, done = pick(env, task, obj='tomato2', loc='table')
        if done:
            print("[Task] tomato2 picked successfully!")
        else:
            print("[Task] Failed to pick tomato2.")
            return

        # Step 9: Place tomato2 on the plate
        print("[Task] Placing tomato2 on the plate.")
        obs, reward, done = place(env, task, obj='tomato2', loc='plate')
        if done:
            print("[Task] tomato2 placed successfully!")
        else:
            print("[Task] Failed to place tomato2.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()