import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate gripper to align with drawer
        print("[Task] Rotating gripper to align with drawer.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 0, 1]))  # Assuming initial orientation is desired

        # Step 2: Move to the side position next to the bottom drawer
        print("[Task] Moving to side position next to the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])

        # Step 3: Move to anchor position to interact with the bottom drawer
        print("[Task] Moving to anchor position for the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, g='gripper', d='bottom', p='anchor_pos_bottom')

        # Step 5: Pull the bottom drawer to access tomatoes
        print("[Task] Pulling the bottom drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='y')

        # Step 6: Pick the first tomato
        print("[Task] Picking up tomato1 from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato1'], loc='table')

        # Step 7: Place the first tomato onto the plate
        print("[Task] Placing tomato1 onto the plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'])

        # Step 8: Pick the second tomato
        print("[Task] Picking up tomato2 from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato2'], loc='table')

        # Step 9: Place the second tomato onto the plate
        print("[Task] Placing tomato2 onto the plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'])

        print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()