import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions based on the provided object list
        bottom_drawer_pos = positions['bottom_anchor_pos']
        rubbish_pos = positions['item3']  # Assuming item3 is rubbish
        bin_pos = positions['bin']

        # === Execute the Plan ===
        # Step 1: Rotate the gripper to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Rotate to 90 degrees
        if done:
            print("[Task] Rotation completed.")
        else:
            print("[Task] Rotation not completed, exiting.")
            return

        # Step 2: Move to the side position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved to side position of the bottom drawer.")
        else:
            print("[Task] Move not completed, exiting.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Moved to anchor position of the bottom drawer.")
        else:
            print("[Task] Move not completed, exiting.")
            return

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick(env, task, target_pos=bottom_drawer_pos)
        if done:
            print("[Task] Picked the bottom drawer.")
        else:
            print("[Task] Pick action not completed, exiting.")
            return

        # Step 5: Pull open the bottom drawer
        obs, reward, done = pull(env, task, pull_distance=0.1)  # Assuming a pull distance
        if done:
            print("[Task] Pulled open the bottom drawer.")
        else:
            print("[Task] Pull action not completed, exiting.")
            return

        # Step 6: Pick the rubbish
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Picked the rubbish.")
        else:
            print("[Task] Pick action not completed, exiting.")
            return

        # Step 7: Place the rubbish into the bin
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Placed the rubbish into the bin.")
        else:
            print("[Task] Place action not completed, exiting.")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()