import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *  # Using the predefined skills directly

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Identify positions of the bottom drawer and rubbish bin
        bottom_drawer_pos = positions['bottom_anchor_pos']
        bin_pos = positions['bin']
        rubbish_pos = positions['rubbish']
        
        # Step 1: Rotate the gripper to the correct orientation
        print("[Task] Rotating the gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi / 4), np.cos(np.pi / 4)]))  # Rotate to 90 degrees
        if done:
            print("[Task] Successfully rotated the gripper!")
        else:
            print("[Task] Could not complete rotation. Aborting.")
            return

        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Moving to the side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos)  # Move to the side position
        if done:
            print("[Task] Successfully moved to side position!")
        else:
            print("[Task] Could not complete move. Aborting.")
            return

        # Step 3: Move to the anchor position to pick the drawer
        print("[Task] Moving to the anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_drawer_pos)  # Move to the anchor position
        if done:
            print("[Task] Successfully moved to anchor position!")
        else:
            print("[Task] Could not complete move. Aborting.")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, 'bottom', bottom_drawer_pos)  # Pick the drawer
        if done:
            print("[Task] Successfully picked the drawer!")
        else:
            print("[Task] Could not complete picking. Aborting.")
            return

        # Step 5: Pull the drawer to unlock the cabinet
        print("[Task] Attempting to unlock the cabinet by pulling the bottom drawer.")
        obs, reward, done = pull(env, task, pull_distance=0.1)  # Assuming pull distance needed
        if done:
            print("[Task] Successfully pulled the bottom drawer!")
        else:
            print("[Task] Could not complete pulling. Aborting.")
            return

        # Step 6: Pick up the rubbish
        print("[Task] Picking up the rubbish.")
        obs, reward, done = pick(env, task, 'rubbish', rubbish_pos)  # Pick the rubbish
        if done:
            print("[Task] Successfully picked up the rubbish!")
        else:
            print("[Task] Could not complete picking. Aborting.")
            return

        # Step 7: Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin.")
        obs, reward, done = place(env, task, target_pos=bin_pos)  # Place in the bin
        if done:
            print("[Task] Successfully placed the rubbish in the bin!")
        else:
            print("[Task] Could not complete placing. Aborting.")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()