import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions for easier reference
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        drawer_pos = positions['bottom_anchor_pos']  # Assuming we only need the bottom drawer
        side_pos = positions['bottom_side_pos']  # Side position of the drawer

        # Step 1: Rotate the gripper (initially at zero degrees)
        print("[Task] Rotating gripper to initial position...")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 0, 1]))
        if done:
            print("[Task] Task ended while rotating the gripper!")
            return

        # Step 2: Move to the side position of the drawer
        print("[Task] Moving to the side position of the drawer...")
        obs, reward, done = move(env, task, target_pos=side_pos)
        if done:
            print("[Task] Task ended while moving to the side position!")
            return

        # Step 3: Move to the anchor position of the drawer
        print("[Task] Moving to the anchor position of the drawer...")
        obs, reward, done = move(env, task, target_pos=drawer_pos)
        if done:
            print("[Task] Task ended while moving to the anchor position!")
            return

        # Step 4: Pick the drawer
        print("[Task] Picking the drawer...")
        obs, reward, done = pick_drawer(env, task, g='gripper', d='bottom', p='anchor-pos-bottom')
        if done:
            print("[Task] Task ended while picking the drawer!")
            return
        
        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer open...")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='y')  # Adjust parameters for pull action
        if done:
            print("[Task] Task ended while pulling the drawer!")
            return

        # Step 6: Pick up the rubbish from the table
        print("[Task] Picking up the rubbish...")
        obs, reward, done = pick(env, task, target_pos=rubbish_pos, approach_distance=0.15)
        if done:
            print("[Task] Task ended while picking up rubbish!")
            return

        # Step 7: Place the rubbish in the bin
        print("[Task] Placing the rubbish in the bin...")
        obs, reward, done = place(env, task, target_pos=bin_pos, approach_distance=0.15)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()