import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define object positions
        rubbish_pos = positions['rubbish']
        bin_pos = positions['bin']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_side_pos = positions['bottom_side_pos']

        # === Execute the Oracle Plan ===
        print("[Task] Starting to open a drawer.")
        
        # Step 1: Rotate to the correct orientation (gripper facing drawer)
        print("[Task] Step 1: Rotate to orientation.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Adjusted quaternion
        if done:
            print("[Task] Task ended after rotation!")
            return
        
        # Step 2: Move to the side position of the bottom drawer
        print("[Task] Step 2: Move to side position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("[Task] Task ended after moving to side position!")
            return
        
        # Step 3: Move to the anchor position of the bottom drawer
        print("[Task] Step 3: Move to anchor position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        if done:
            print("[Task] Task ended after moving to anchor position!")
            return
        
        # Step 4: Pick the drawer
        print("[Task] Step 4: Attempting to pick the drawer.")
        obs, reward, done = pick(env, task, 'bottom', 'bottom_anchor_pos')  # Updated to use pick action
        if done:
            print("[Task] Task ended after picking drawer!")
            return
        
        # Step 5: Pull the drawer to open it
        print("[Task] Step 5: Pull the drawer to open it.")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')  # Distance to pull
        if done:
            print("[Task] Task ended after pulling drawer!")
            return
        
        # Step 6: Pick the rubbish from the table
        print("[Task] Step 6: Pick the rubbish.")
        obs, reward, done = pick(env, task, 'rubbish', 'table')  
        if done:
            print("[Task] Task ended after picking rubbish!")
            return
        
        # Step 7: Place the rubbish in the bin
        print("[Task] Step 7: Place the rubbish in the bin.")
        obs, reward, done = place(env, task, 'rubbish', 'bin')  
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
            return
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()