import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate the gripper to 90 degrees
        print("Step 1: Rotating the gripper to 90 degrees.")
        rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Adjust quaternion for 90 degrees

        # Step 2: Move to side position of the bottom drawer
        print("Step 2: Moving to side position of the bottom drawer.")
        move(env, task, positions['bottom_side_pos'])

        # Step 3: Move to anchor position of the bottom drawer
        print("Step 3: Moving to anchor position of the bottom drawer.")
        move(env, task, positions['bottom_anchor_pos'])

        # Step 4: Pick the bottom drawer
        print("Step 4: Picking the bottom drawer.")
        pick(env, task, 'bottom', positions['bottom_anchor_pos'])

        # Step 5: Pull the bottom drawer to unlock it
        print("Step 5: Pulling the bottom drawer.")
        pull(env, task, pull_distance=0.1)  # Adjust according to task specifications.

        # Step 6: Pick up the rubbish
        print("Step 6: Picking the rubbish.")
        pick(env, task, 'rubbish', positions['rubbish'])

        # Step 7: Place the rubbish in the bin
        print("Step 7: Placing the rubbish in the bin.")
        place(env, task, 'rubbish', positions['bin'])

    except Exception as e:
        print("[Error] An error occurred: ", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()