import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define object positions
        bottom_anchor_pos = positions['bottom_anchor_pos']
        bottom_side_pos = positions['bottom_side_pos']
        rubbish_pos = positions['rubbish']  # Position of rubbish
        bin_pos = positions['bin']            # Position of bin

        # === Execute Steps from Oracle Plan ===
        # Step 1: Rotate the gripper to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100, threshold=0.05, timeout=10.0)
        if done:
            print("[Task] Task ended during rotation (Step 1)!")
            return

        # Step 2: Move gripper to side position bottom
        obs, reward, done = move(env, task, target_pos=bottom_side_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during movement to side position (Step 2)!")
            return

        # Step 3: Move gripper to anchor position bottom
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos, max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during movement to anchor position (Step 3)!")
            return

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick(env, task, 'gripper', 'bottom', bottom_anchor_pos)
        if done:
            print("[Task] Task ended during picking drawer (Step 4)!")
            return

        # Step 5: Pull the bottom drawer open
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='z', max_steps=100, threshold=0.01, timeout=10.0)
        if done:
            print("[Task] Task ended during pulling drawer (Step 5)!")
            return
        
        # Step 6: Pick up the rubbish
        obs, reward, done = pick(env, task, 'rubbish', rubbish_pos)
        if done:
            print("[Task] Task ended during picking rubbish (Step 6)!")
            return
        
        # Step 7: Place the rubbish into the bin
        obs, reward, done = place(env, task, 'rubbish', bin_pos)
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False).")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()