import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate gripper to ninety degrees
        print("[Task] Rotating to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Rotate using appropriate quaternion
        if done:
            print("[Task] Task ended during rotation!")
            return

        # Step 2: Move gripper to side position near the bottom drawer
        print("[Task] Moving to side position near the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended during move!")
            return

        # Step 3: Move gripper to anchor position for the bottom drawer
        print("[Task] Moving to anchor position of bottom drawer.")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended during move!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick_drawer(env, task, 'gripper', 'bottom', 'bottom_anchor_pos')
        if done:
            print("[Task] Task ended during picking drawer!")
            return
        
        # Step 5: Pull the bottom drawer open
        print("[Task] Pulling the bottom drawer to open it.")
        obs, reward, done = pull(env, task, pull_distance=0.2)  # Pull distance of 0.2
        if done:
            print("[Task] Task ended during pulling drawer!")
            return
        
        # Step 6: Pick the rubbish from the table
        print("[Task] Picking up the rubbish from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['rubbish'], approach_distance=0.15)
        if done:
            print("[Task] Task ended during picking rubbish!")
            return

        # Step 7: Place the rubbish into the bin
        print("[Task] Placing the rubbish into the bin.")
        obs, reward, done = place(env, task, target_pos=positions['bin'], approach_distance=0.15)
        if done:
            print("[Task] Task ended during placing rubbish!")
            return

        print("[Task] All tasks completed successfully!")
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()