import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        
        # Define the drawer object references
        bottom_drawer = 'bottom'

        # Step 1: Rotate the gripper from zero to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, np.sin(np.pi/4), np.cos(np.pi/4)]))  # Rotating 90 degrees
        if done:
            print("[Task] Rotated the gripper!")
        else:
            print("[Task] Did not complete rotating the gripper.")

        # Step 2: Move to the side position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Moved to side position of the bottom drawer!")
        else:
            print("[Task] Did not complete moving to side position of the bottom drawer.")

        # Step 3: Move to the anchor position of the bottom drawer
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Moved to anchor position of the bottom drawer!")
        else:
            print("[Task] Did not complete moving to anchor position of the bottom drawer.")

        # Step 4: Pick the bottom drawer
        obs, reward, done = pick(env, task, 'gripper', bottom_drawer, positions['bottom_anchor_pos'])  # Pick the drawer at anchor position
        if done:
            print("[Task] Picked the bottom drawer!")
        else:
            print("[Task] Did not complete picking the drawer.")

        # Step 5: Pull the bottom drawer open
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')  # Pull the drawer
        if done:
            print("[Task] Pulled open the bottom drawer!")
        else:
            print("[Task] Did not complete pulling the drawer.")

        # Step 6: Pick the first tomato
        obs, reward, done = pick(env, task, target_pos=positions['tomato1'], approach_distance=0.15)
        if done:
            print("[Task] Picked the first tomato!")
        else:
            print("[Task] Did not complete picking the first tomato.")

        # Step 7: Place the first tomato onto the plate
        obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.15)
        if done:
            print("[Task] Placed the first tomato on the plate!")
        else:
            print("[Task] Did not complete placing the first tomato.")

        # Step 8: Pick the second tomato
        obs, reward, done = pick(env, task, target_pos=positions['tomato2'], approach_distance=0.15)
        if done:
            print("[Task] Picked the second tomato!")
        else:
            print("[Task] Did not complete picking the second tomato.")

        # Step 9: Place the second tomato onto the plate
        obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.15)
        if done:
            print("[Task] Placed the second tomato on the plate!")
        else:
            print("[Task] Did not complete placing the second tomato.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()