import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, pull, rotate, move

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Unlock the cabinet by opening an accessible drawer
        print("[Task] Opening an accessible drawer to unlock the cabinet...")
        obs, reward, done = rotate(env, task, target_quat='ninety_deg')  # Rotate gripper to align
        if done:
            print("[Task] Gripper rotated successfully!")

        print("[Task] Moving gripper to side position for drawer opening...")
        obs, reward, done = move(env, task, target_pos='bottom_side_pos')
        if done:
            print("[Task] Moved to side position successfully!")

        print("[Task] Moving gripper to anchor position for drawer...")
        obs, reward, done = move(env, task, target_pos='bottom_anchor_pos')
        if done:
            print("[Task] Moved to anchor position successfully!")

        print("[Task] Picking drawer...")
        obs, reward, done = pick(env, task, obj='bottom', loc='bottom_anchor_pos')
        if done:
            print("[Task] Drawer picked successfully!")

        print("[Task] Pulling the drawer to open...")
        obs, reward, done = pull(env, task, pull_distance=0.5, pull_axis='-x')
        if done:
            print("[Task] Drawer opened successfully!")
        else:
            print("[Task] Failed to open drawer.")
            return  # Exit if drawer could not be opened

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 6: Picking first tomato
        print("[Task] Picking up tomato1...")
        obs, reward, done = pick(env, task, obj='tomato1', loc=positions['tomato1'])
        if done:
            print("[Task] Picked tomato1 successfully!")
            print("[Task] Placing tomato1 on the plate...")
            obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.15)
            if done:
                print("[Task] Placed tomato1 on the plate successfully!")
            else:
                print("[Task] Failed to place tomato1.")
                return  # Exit if placing failed
        
        # Step 8: Picking second tomato
        print("[Task] Picking up tomato2...")
        obs, reward, done = pick(env, task, obj='tomato2', loc=positions['tomato2'])
        if done:
            print("[Task] Picked tomato2 successfully!")
            print("[Task] Placing tomato2 on the plate...")
            obs, reward, done = place(env, task, target_pos=positions['plate'], approach_distance=0.15)
            if done:
                print("[Task] Placed tomato2 on the plate successfully!")
            else:
                print("[Task] Failed to place tomato2.")
        else:
            print("[Task] Failed to pick tomato2.")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()