import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_anchor_pos = positions['bottom_anchor_pos']
        side_pos_bottom = positions['bottom_side_pos']
        plate_pos = positions['plate']

        # Step 1: Rotate gripper to 90 degrees
        print("[Task] Rotating gripper to 90 degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100)
        if done:
            print("[Task] Successfully rotated the gripper!")
        else:
            print("[Task] Failed to rotate the gripper!")
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Task] Moving to side position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom, max_steps=100)
        if done:
            print("[Task] Successfully moved the gripper!")
        else:
            print("[Task] Failed to move the gripper!")
            return

        # Step 3: Move gripper to the anchor position of the bottom drawer
        print("[Task] Moving to anchor position of the bottom drawer.")
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos, max_steps=100)
        if done:
            print("[Task] Successfully moved to anchor position!")
        else:
            print("[Task] Failed to move to anchor position!")
            return

        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=bottom_anchor_pos, approach_distance=0.15)
        if done:
            print("[Task] Successfully picked the drawer!")
        else:
            print("[Task] Failed to pick the drawer!")
            return

        # Step 5: Pull the drawer open
        print("[Task] Pulling the drawer open.")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Successfully opened the drawer!")
        else:
            print("[Task] Failed to pull the drawer!")
            return

        # Step 6: Pick tomato1 from the table
        print("[Task] Picking tomato1 from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato1'], approach_distance=0.15)
        if done:
            print("[Task] Successfully picked tomato1!")
        else:
            print("[Task] Failed to pick tomato1!")
            return

        # Step 7: Place tomato1 onto the plate
        print("[Task] Placing tomato1 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15)
        if done:
            print("[Task] Successfully placed tomato1!")
        else:
            print("[Task] Failed to place tomato1!")
            return

        # Step 8: Pick tomato2 from the table
        print("[Task] Picking tomato2 from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato2'], approach_distance=0.15)
        if done:
            print("[Task] Successfully picked tomato2!")
        else:
            print("[Task] Failed to pick tomato2!")
            return

        # Step 9: Place tomato2 onto the plate
        print("[Task] Placing tomato2 onto the plate.")
        obs, reward, done = place(env, task, target_pos=plate_pos, approach_distance=0.15)
        if done:
            print("[Task] Successfully placed tomato2!")
        else:
            print("[Task] Failed to place tomato2!")
            return

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()