import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Define specific positions
        plate_pos = positions['plate']
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        side_pos_bottom = positions['bottom_side_pos']

        # === Execute Plan Steps ===

        # Step 1: Rotate gripper to ninety degrees
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]), max_steps=100)
        if done:
            print("[Run] Task ended after rotating the gripper!")
            return

        # Step 2: Move to side position for bottom drawer
        obs, reward, done = move(env, task, target_pos=side_pos_bottom, max_steps=100)
        if done:
            print("[Run] Task ended after moving to side position!")
            return

        # Step 3: Move to anchor position for bottom drawer
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos, max_steps=100)
        if done:
            print("[Run] Task ended after moving to anchor position!")
            return
        
        # Step 4: Pick the bottom drawer
        obs, reward, done = pick_drawer(env, task, "gripper", "bottom", bottom_anchor_pos)
        if done:
            print("[Run] Task ended after picking the drawer!")
            return

        # Step 5: Pull the bottom drawer open
        obs, reward, done = pull(env, task, pull_distance=0.1)  # Specify distance here if needed
        if done:
            print("[Run] Task ended after pulling the drawer!")
            return

        # Step 6: Pick the first tomato
        obs, reward, done = pick(env, task, "tomato1", "table")
        if done:
            print("[Run] Task ended after picking first tomato!")
            return

        # Step 7: Place the first tomato on the plate
        obs, reward, done = place(env, task, "tomato1", "plate")
        if done:
            print("[Run] Task ended after placing first tomato!")
            return

        # Step 8: Pick the second tomato
        obs, reward, done = pick(env, task, "tomato2", "table")
        if done:
            print("[Run] Task ended after picking second tomato!")
            return

        # Step 9: Place the second tomato on the plate
        obs, reward, done = place(env, task, "tomato2", "plate")
        if done:
            print("[Run] Task completed successfully with second tomato!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()