import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate the gripper to ninety degrees to align for moving
        print("[Task] Rotating gripper to ninety degrees.")
        obs, reward, done = rotate(env, task, target_quat=np.array([0, 0, 1, 0]))  # Quaternion for 90 degrees
        if done:
            print("[Task] Task ended during rotation!")
            return
        
        # Step 2: Move the gripper to side position of the bottom drawer
        print("[Task] Moving gripper to side position (bottom).")
        obs, reward, done = move(env, task, target_pos=positions['bottom_side_pos'])
        if done:
            print("[Task] Task ended during moving to side position!")
            return
        
        # Step 3: Move to anchor position (bottom drawer)
        print("[Task] Moving gripper to anchor position (bottom).")
        obs, reward, done = move(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended during moving to anchor position!")
            return
        
        # Step 4: Pick the bottom drawer
        print("[Task] Picking the bottom drawer.")
        obs, reward, done = pick(env, task, target_pos=positions['bottom_anchor_pos'])
        if done:
            print("[Task] Task ended while picking the drawer!")
            return
        
        # Step 5: Pull the drawer to open it
        print("[Task] Pulling the bottom drawer to open it.")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='x')
        if done:
            print("[Task] Task ended while pulling the drawer!")
            return
        
        # Step 6: Pick the first tomato from the table
        print("[Task] Picking the first tomato from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato1'])
        if done:
            print("[Task] Task ended while picking the first tomato!")
            return
        
        # Step 7: Place the first tomato onto the plate
        print("[Task] Placing the first tomato onto the plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'])
        if done:
            print("[Task] Task ended while placing the first tomato!")
            return
        
        # Step 8: Pick the second tomato from the table
        print("[Task] Picking the second tomato from the table.")
        obs, reward, done = pick(env, task, target_pos=positions['tomato2'])
        if done:
            print("[Task] Task ended while picking the second tomato!")
            return
        
        # Step 9: Place the second tomato onto the plate
        print("[Task] Placing the second tomato onto the plate.")
        obs, reward, done = place(env, task, target_pos=positions['plate'])
        if done:
            print("[Task] Task completed successfully!")
            return
        
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()