import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the disposal task according to the oracle plan.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Ensure required objects are available
        for name in ['tomato1', 'tomato2', 'plate']:
            if name not in positions:
                print(f"[Error] Object '{name}' not found in positions.")
                return

        plate_pos = np.array(positions['plate'])
        tomato_names = ['tomato1', 'tomato2']

        # === Execute Oracle Plan ===
        for tomato in tomato_names:
            obj_pos = np.array(positions[tomato])

            # Step: pick tomato from table
            print(f"[Task] Picking {tomato} from table at: {obj_pos}")
            try:
                obs, reward, done = pick(
                                env,
                                task,
                                target_pos=obj_pos,
                                approach_distance=0.15,
                                max_steps=100,
                                threshold=0.01,
                                approach_axis='z',
                                timeout=10.0
                            )
            except Exception as e:
                print(f"[Error] Exception during pick({tomato}): {e}")
                break
            if done:
                print(f"[Task] Episode ended after pick({tomato}). Reward: {reward}")
                return

            # Step: place tomato onto plate
            print(f"[Task] Placing {tomato} on plate at: {plate_pos}")
            try:
                obs, reward, done = place(
                    env,
                    task,
                    target_pos=plate_pos,
                    approach_distance=0.15,
                    max_steps=100,
                    threshold=0.01,
                    approach_axis='z',
                    timeout=10.0
                )
            except Exception as e:
                print(f"[Error] Exception during place({tomato}): {e}")
                break
            if done:
                print(f"[Task] Episode ended after place({tomato}). Reward: {reward}")
                return

        # === Final Status ===
        print("[Task] Completed all pick-and-place steps.")
        print(f"[Task] Final reward: {reward}, done flag: {done}")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()