import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        try:
            switch_pos = np.array(positions['switch'])
            tomato1_pos = np.array(positions['tomato1'])
            tomato2_pos = np.array(positions['tomato2'])
            plate_pos = np.array(positions['plate'])
        except KeyError as e:
            print(f"[Error] Position for object {e} not found in positions dict.")
            return

        # === Execute Oracle Plan Steps ===

        # Step 1: press switch
        print(f"[Task] Step 1: Pressing 'switch' at position {switch_pos}")
        obs, reward, done = press(
            env,
            task,
            target_pos=switch_pos
        )
        if done:
            print("[Task] Task ended prematurely after press(switch). Reward:", reward)
            return

        # Step 2: pick tomato1 from table
        print(f"[Task] Step 2: Picking 'tomato1' at position {tomato1_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos
        )
        if done:
            print("[Task] Task ended prematurely after pick(tomato1). Reward:", reward)
            return

        # Step 3: place tomato1 on plate
        print(f"[Task] Step 3: Placing 'tomato1' on plate at position {plate_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos
        )
        if done:
            print("[Task] Task ended prematurely after place(tomato1). Reward:", reward)
            return

        # Step 4: pick tomato2 from table
        print(f"[Task] Step 4: Picking 'tomato2' at position {tomato2_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos
        )
        if done:
            print("[Task] Task ended prematurely after pick(tomato2). Reward:", reward)
            return

        # Step 5: place tomato2 on plate
        print(f"[Task] Step 5: Placing 'tomato2' on plate at position {plate_pos}")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos
        )
        if done:
            print("[Task] Task ended prematurely after place(tomato2). Reward:", reward)
            return

        # Final result
        if not done:
            print("[Task] Completed all plan steps. Final reward:", reward)
        else:
            print("[Task] Done flag signaled at end of plan. Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()