import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Execute the oracle plan: close gripper, press switch, pick and place two tomatoes.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset to initial state
        descriptions, obs = task.reset()
        # Initialize video recording if desired
        init_video_writers(obs)
        # Wrap step functions for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # === Retrieve Object Positions ===
        positions = get_object_positions()
        try:
            switch_pos = np.array(positions['switch'])
            tomato1_pos = np.array(positions['tomato1'])
            tomato2_pos = np.array(positions['tomato2'])
            plate_pos = np.array(positions['plate'])
        except KeyError as e:
            raise KeyError(f"Required object position not found: {e}")
        
        # === Oracle Plan Execution ===
        # Step 1: close_gripper
        print("[Task] Step 1: Closing gripper")
        obs, reward, done = close_gripper(env, task)
        if done:
            print("[Task] Task ended after closing gripper. Reward:", reward)
            return
        
        # Step 2: press switch
        print("[Task] Step 2: Pressing switch at:", switch_pos)
        obs, reward, done = press(env, task, target_pos=switch_pos)
        if done:
            print("[Task] Task ended after pressing switch. Reward:", reward)
            return
        
        # Step 3: pick tomato1
        print("[Task] Step 3: Picking tomato1 at:", tomato1_pos)
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Task ended after picking tomato1. Reward:", reward)
            return
        
        # Step 4: place tomato1 onto plate
        print("[Task] Step 4: Placing tomato1 onto plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing tomato1. Reward:", reward)
            return
        
        # Step 5: pick tomato2
        print("[Task] Step 5: Picking tomato2 at:", tomato2_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Task ended after picking tomato2. Reward:", reward)
            return
        
        # Step 6: place tomato2 onto plate
        print("[Task] Step 6: Placing tomato2 onto plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Task ended after placing tomato2. Reward:", reward)
            return
        
        # Success
        print("[Task] Successfully placed both tomatoes onto the plate. Final reward:", reward)
    
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()