import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers
        init_video_writers(obs)

        # Wrap the task step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expect keys: 'plate', 'item1', 'item2', 'item3', 'switch', 'bin'

        # === Task Plan ===

        # 1) Close gripper so we can press the switch
        try:
            print("[Task] Closing gripper to prepare for pressing switch")
            obs, reward, done = close_gripper(env, task)
            if done:
                print("[Task] Task ended during close_gripper. Reward:", reward)
                return
        except Exception as e:
            print(f"[Task] Error during close_gripper: {e}")
            return

        # 2) Press the switch to turn off 'dark' and enable picks/places
        try:
            switch_pos = positions['switch']
            print("[Task] Pressing switch at position:", switch_pos)
            obs, reward, done = press(env, task, target_pos=np.array(switch_pos))
            if done:
                print("[Task] Task ended during press. Reward:", reward)
                return
        except Exception as e:
            print(f"[Task] Error during press: {e}")
            return

        # 3) For each rubbish item (item1, item2, item3), pick and place into bin
        rubbish_items = ['item1', 'item2', 'item3']
        bin_pos = positions['bin']
        for item in rubbish_items:
            try:
                item_pos = positions[item]
                print(f"[Task] Picking up {item} at position: {item_pos}")
                obs, reward, done = pick(env, task, target_pos=np.array(item_pos))
                if done:
                    print(f"[Task] Task ended during pick of {item}. Reward:", reward)
                    return
            except Exception as e:
                print(f"[Task] Error during pick of {item}: {e}")
                return

            try:
                print(f"[Task] Placing {item} into bin at position: {bin_pos}")
                obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
                if done:
                    print(f"[Task] Task ended during place of {item}. Reward:", reward)
                    return
            except Exception as e:
                print(f"[Task] Error during place of {item}: {e}")
                return

        # 4) Task complete
        print("[Task] All rubbish items have been moved to the bin.")
        print("[Task] Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()