# run_skeleton_task.py (Fully Generic Skeleton)
#
# Description:
#  - This code is a skeleton that shows how to set up an environment, retrieve
#    object positions, and execute a plan without being tied to a specific skill
#    like pick or place.
#  - Use this as a base for creating your own task logic.
#  - The sections that are task-dependent (e.g., manipulation steps) are marked
#    with TODO or placeholders.

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        # positions is a dict mapping object names to their 3D positions
        positions = get_object_positions()

        # === Oracle Plan Execution ===
        # Specification: pick rubbish from the table, then place it into the bin.

        # Step 1: Pick up the rubbish
        if 'rubbish' not in positions:
            raise KeyError("Object 'rubbish' not found in positions.")
        rubbish_pos = positions['rubbish']
        print("[Step 1] Picking up 'rubbish' from table at position:", rubbish_pos)
        [Frozen Code Start]
        obs, reward, done = pick(
                env,
                task,
                target_pos=rubbish_pos,
                approach_distance=0.15,
                max_steps=100,
                threshold=0.01,
                approach_axis='z',
                timeout=10.0
            )
        [Frozen Code End]
        if done:
            print("[Task] Terminated early during pick. Reward:", reward)
            return

        # Step 2: Place the rubbish into the bin
        if 'bin' not in positions:
            raise KeyError("Location 'bin' not found in positions.")
        bin_pos = positions['bin']
        print("[Step 2] Placing 'rubbish' into 'bin' at position:", bin_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=bin_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Completed successfully! Final Reward:", reward)
        else:
            print("[Task] Plan executed but 'done' flag is False. Reward:", reward)

    except Exception as e:
        print("[Error] Exception during task execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()