from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np

def run_skeleton_task():
    '''Generic skeleton for running the specified drawer‐open and disposal task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing the simulation
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Drawer positions
        side_bottom = np.array(positions['bottom_side_pos'])
        anchor_bottom = np.array(positions['bottom_anchor_pos'])
        # Rubbish and bin positions
        rubbish_pos = np.array(positions.get('rubbish', positions.get('item3')))
        bin_pos = np.array(positions['bin'])

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero to ninety degrees
        # Use quaternion for 90° around X: [x, y, z, w]
        target_quat = np.array([0.7071, 0.0, 0.0, 0.7071])
        print("[Task] Step 1: Rotating gripper to 90°:", target_quat)
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        if done:
            print("[Task] Early termination after rotate. Reward:", reward)
            return

        # Step 2: Move gripper from its current (nowhere) position to drawer side
        print("[Task] Step 2: Moving to drawer side position:", side_bottom)
        obs, reward, done = move(env, task, target_pos=side_bottom)
        if done:
            print("[Task] Early termination after move to side. Reward:", reward)
            return

        # Step 3: Move gripper to drawer anchor position
        print("[Task] Step 3: Moving to drawer anchor position:", anchor_bottom)
        obs, reward, done = move(env, task, target_pos=anchor_bottom)
        if done:
            print("[Task] Early termination after move to anchor. Reward:", reward)
            return

        # Step 4: Pull the bottom drawer open
        pull_distance = 0.1  # meters
        print(f"[Task] Step 4: Pulling drawer by {pull_distance}m along +X axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Early termination after pull. Reward:", reward)
            return

        # Step 5: Pick up the rubbish from the table
        print("[Task] Step 5: Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(env, task, target_pos=rubbish_pos)
        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return

        # Step 6: Place the rubbish into the bin
        print("[Task] Step 6: Placing rubbish into bin at:", bin_pos)
        obs, reward, done = place(env, task, target_pos=bin_pos)
        if done:
            print("[Task] Task completed successfully: rubbish placed in bin. Reward:", reward)
        else:
            print("[Task] Task ended without full completion. Reward:", reward)

    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()