import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Task Plan ===

        # 1) Rotate gripper to 90 degrees around the Z axis
        print("[Task] Rotating gripper to 90 degrees")
        # quaternion for 90 deg about Z: [x, y, z, w]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Early termination after rotate. Reward:", reward)
            return

        # 2) Move to the side position of the bottom drawer
        side_pos = positions['bottom_side_pos']
        print(f"[Task] Moving to bottom drawer side position at {side_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(side_pos))
        if done:
            print("[Task] Early termination after move-to-side. Reward:", reward)
            return

        # 3) Move to the anchor position of the bottom drawer
        anchor_pos = positions['bottom_anchor_pos']
        print(f"[Task] Moving to bottom drawer anchor position at {anchor_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(anchor_pos))
        if done:
            print("[Task] Early termination after move-to-anchor. Reward:", reward)
            return

        # 4) Pull to open the drawer
        pull_distance = 0.1  # meters
        pull_axis = 'x'
        print(f"[Task] Pulling drawer by {pull_distance}m along {pull_axis} axis")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Task] Early termination after pull. Reward:", reward)
            return

        # 5) Pick up the rubbish (assumed to be 'item3')
        rubbish_pos = positions['item3']
        print(f"[Task] Picking up rubbish at {rubbish_pos}")
        obs, reward, done = pick(env, task, target_pos=np.array(rubbish_pos))
        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return

        # 6) Place the rubbish in the bin
        bin_pos = positions['bin']
        print(f"[Task] Placing rubbish in bin at {bin_pos}")
        obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task not completed yet (done=False). Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()