import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, rotate, pull, pick, place
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()

        # Initialize video recording if needed
        init_video_writers(obs)
        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve all object positions
        positions = get_object_positions()

        # Step 1: Move to the side position of the bottom drawer
        bottom_side_pos = positions['bottom_side_pos']
        print("[Task] Step 1: Moving to bottom side position:", bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=np.array(bottom_side_pos))
        if done:
            print("[Task] Ended after move. Reward:", reward)
            return

        # Step 2: Rotate gripper by 90 degrees (around z-axis)
        print("[Task] Step 2: Rotating gripper to ninety_deg orientation")
        # Quaternion for a 90° rotation about the z-axis: [x, y, z, w]
        ninety_quat = np.array([0.0, 0.0, np.sqrt(2)/2, np.sqrt(2)/2])
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Ended after rotate. Reward:", reward)
            return

        # Step 3: Move to the anchor position of the bottom drawer
        bottom_anchor_pos = positions['bottom_anchor_pos']
        print("[Task] Step 3: Moving to bottom anchor position:", bottom_anchor_pos)
        obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor_pos))
        if done:
            print("[Task] Ended after move to anchor. Reward:", reward)
            return

        # Step 4: Pull the drawer open
        print("[Task] Step 4: Pulling drawer open")
        # Choose a pull distance sufficient to open the drawer
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended after pull. Reward:", reward)
            return

        # Step 5: Pick up the rubbish
        rubbish_pos = positions['rubbish']
        print("[Task] Step 5: Picking up rubbish at:", rubbish_pos)
        obs, reward, done = pick(env, task, target_pos=np.array(rubbish_pos))
        if done:
            print("[Task] Ended after pick. Reward:", reward)
            return

        # Step 6: Place the rubbish into the bin
        bin_pos = positions['bin']
        print("[Task] Step 6: Placing rubbish in bin at:", bin_pos)
        obs, reward, done = place(env, task, target_pos=np.array(bin_pos))
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Completed placement but done flag is False. Reward:", reward)

    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()