import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Concrete implementation of the plan to open a drawer and discard rubbish.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to initial state
        descriptions, obs = task.reset()

        # Initialize video recording if desired
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Map plan parameters to actual numpy arrays
        side_pos_bottom   = np.array(positions['bottom_side_pos'])
        anchor_pos_bottom = np.array(positions['bottom_anchor_pos'])
        rubbish_pos       = np.array(positions['rubbish'])
        bin_pos           = np.array(positions['bin'])

        # Compute quaternions for zero and ninety degrees (about z-axis)
        zero_quat   = R.from_euler('xyz', [0, 0, 0], degrees=True).as_quat()
        ninety_quat = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()

        # === Execute Oracle Plan ===

        # Step 1: rotate gripper to ninety degrees
        print("[Step 1] Rotating gripper to ninety degrees")
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("Early termination after rotate.")
            return

        # Step 2: move to bottom drawer side position
        print("[Step 2] Moving gripper to bottom side position")
        obs, reward, done = move(env, task, side_pos_bottom)
        if done:
            print("Early termination after move to side_pos_bottom.")
            return

        # Step 3: move to bottom drawer anchor position
        print("[Step 3] Moving gripper to bottom anchor position")
        obs, reward, done = move(env, task, anchor_pos_bottom)
        if done:
            print("Early termination after move to anchor_pos_bottom.")
            return

        # Step 4: pull to open the bottom drawer
        print("[Step 4] Pulling to open the bottom drawer")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("Early termination after pull.")
            return

        # Step 5: move to rubbish object
        print("[Step 5] Moving gripper to rubbish")
        obs, reward, done = move(env, task, rubbish_pos)
        if done:
            print("Early termination after move to rubbish.")
            return

        # Step 6: pick up the rubbish
        print("[Step 6] Picking up rubbish")
        obs, reward, done = pick(env, task, rubbish_pos)
        if done:
            print("Early termination after pick.")
            return

        # Step 7: move to trash bin
        print("[Step 7] Moving gripper with rubbish to bin")
        obs, reward, done = move(env, task, bin_pos)
        if done:
            print("Early termination after move to bin.")
            return

        # Step 8: place rubbish into bin
        print("[Step 8] Placing rubbish into bin")
        obs, reward, done = place(env, task, bin_pos)
        if done:
            print("Task completed successfully! Final reward:", reward)
        else:
            print("Task finished but done flag is False. Reward:", reward)

    except Exception as e:
        print("Exception encountered during task execution:", e)

    finally:
        # Ensure we always shut down the environment
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()