import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Drawer positions
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        # Plate position
        plate_pos = positions['plate']
        # Tomato names (we only have two to place)
        tomato_names = ['tomato1', 'tomato2']

        # === Task Plan Execution ===

        # Step 1: Determine current gripper orientation
        obs = task.get_observation()
        current_quat = normalize_quaternion(obs.gripper_pose[3:7])

        # [Frozen Code Start]
        obs, reward, done = rotate(env, task, current_quat)
        obs, reward, done = move(env, task, target_pos=np.array(bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        # [Frozen Code End]

        if done:
            print("Task ended during initial drawer operations.")
            return

        # Steps 5-8: Pick each tomato and place it on the plate
        step_counter = 5
        for name in tomato_names:
            item_pos = positions[name]
            print(f"[Step {step_counter}] Picking up {name} at position {item_pos}")
            obs, reward, done = pick(env, task, target_pos=np.array(item_pos))
            if done:
                print(f"Task ended during pick of {name}.")
                return

            print(f"[Step {step_counter+1}] Placing {name} on the plate at position {plate_pos}")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
            if done:
                print(f"Task ended during place of {name}.")
                return

            step_counter += 2

        print("All tomatoes placed on the plate. Task complete!")

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()