import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the drawer-opening and tomato-placement task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        bottom_side = positions.get('bottom_side_pos')
        tomato1_pos = positions.get('item1')
        tomato2_pos = positions.get('item2')
        plate_pos   = positions.get('plate')

        if bottom_side is None or tomato1_pos is None or tomato2_pos is None or plate_pos is None:
            raise ValueError("One or more required object positions not found in positions dict.")

        # === Task Plan Execution ===

        # Step 1: Move gripper to side of the bottom drawer handle
        print("[Task] Step 1: Moving to bottom drawer side at:", bottom_side)
        [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=np.array(bottom_side)
        [Frozen Code End]
        if done:
            print("[Task] Task ended prematurely after move.")
            return

        # Step 2: Pull the drawer open
        print("[Task] Step 2: Pulling drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.05, pull_axis='x')
        if done:
            print("[Task] Task ended prematurely after pull.")
            return

        # Step 3: Pick the first tomato
        print("[Task] Step 3: Picking first tomato at:", tomato1_pos)
        obs, reward, done = pick(env, task, target_pos=np.array(tomato1_pos))
        if done:
            print("[Task] Task ended prematurely after first pick.")
            return

        # Step 4: Place first tomato onto the plate
        print("[Task] Step 4: Placing first tomato on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Task] Task ended prematurely after first place.")
            return

        # Step 5: Pick the second tomato
        print("[Task] Step 5: Picking second tomato at:", tomato2_pos)
        obs, reward, done = pick(env, task, target_pos=np.array(tomato2_pos))
        if done:
            print("[Task] Task ended prematurely after second pick.")
            return

        # Step 6: Place second tomato onto the plate
        print("[Task] Step 6: Placing second tomato on plate at:", plate_pos)
        obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
        if done:
            print("[Task] Task ended prematurely after second place.")
            return

        # Final check
        print("[Task] Completed all steps. Final reward:", reward, "Done flag:", done)

    except Exception as e:
        print(f"[Task] Exception encountered: {e}")
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()