import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running the drawer and tomato task.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper to 90° around Z-axis for side approach
        print("[Step 1] Rotating gripper to 90° around Z-axis")
        target_quat = np.array([0.0, 0.0, np.sqrt(2)/2, np.sqrt(2)/2])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended during rotation.")
            return

        # Step 2: Move to bottom drawer side position
        print("[Step 2] Moving to bottom drawer side position")
        side_pos = positions['bottom_side_pos']
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Ended during move to side.")
            return

        # Step 3: Move from side to anchor position on bottom drawer
        print("[Step 3] Moving to bottom drawer anchor position")
        anchor_pos = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Ended during move to anchor.")
            return

        # Step 4: Pull the drawer open
        print("[Step 4] Pulling the bottom drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Ended during pull.")
            return

        # Step 5: Pick the first tomato
        print("[Step 5] Picking first tomato")
        tomato1_pos = positions['tomato1']
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Ended during pick of tomato1.")
            return

        # Step 6: Place the first tomato on the plate
        print("[Step 6] Placing first tomato on plate")
        plate_pos = positions['plate']
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ended during place of tomato1.")
            return

        # Step 7: Pick the second tomato
        print("[Step 7] Picking second tomato")
        tomato2_pos = positions['tomato2']
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Ended during pick of tomato2.")
            return

        # Step 8: Place the second tomato on the plate
        print("[Step 8] Placing second tomato on plate")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ended during place of tomato2.")
            return

        print("[Task] Completed successfully. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()