import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        
        # Step 1: Rotate gripper to 90° about Z so it can align with drawer side
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("[Task] Step 1: Rotating gripper to 90° around Z axis.")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Ended after rotate.")
            return

        # Step 2: Move to the bottom drawer anchor position
        drawer_anchor_pos = positions['bottom_anchor_pos']
        print(f"[Task] Step 2: Moving to bottom drawer anchor pos: {drawer_anchor_pos}")
        obs, reward, done = move(env, task, drawer_anchor_pos)
        if done:
            print("[Task] Ended after move.")
            return

        # Step 3: Pull to open the drawer
        pull_distance = 0.1
        pull_axis = 'x'
        print(f"[Task] Step 3: Pulling drawer open (distance={pull_distance}, axis={pull_axis}).")
        obs, reward, done = pull(env, task, pull_distance, pull_axis)
        if done:
            print("[Task] Ended after pull.")
            return

        # Step 4: Pick tomato1 from the table
        tomato1_pos = positions['tomato1']
        print(f"[Task] Step 4: Picking tomato1 at {tomato1_pos}")
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        if done:
            print("[Task] Ended after picking tomato1.")
            return

        # Step 5: Place tomato1 on the plate
        plate_pos = positions['plate']
        print(f"[Task] Step 5: Placing tomato1 on plate at {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Ended after placing tomato1.")
            return

        # Step 6: Pick tomato2 from the table
        tomato2_pos = positions['tomato2']
        print(f"[Task] Step 6: Picking tomato2 at {tomato2_pos}")
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        if done:
            print("[Task] Ended after picking tomato2.")
            return

        # Step 7: Place tomato2 on the plate
        print(f"[Task] Step 7: Placing tomato2 on plate at {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Completed successfully with reward:", reward)
        else:
            print("[Task] All steps executed. done flag:", done, "reward:", reward)

    finally:
        # Ensure proper shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()