import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Assign relevant object positions
        drawer_side_pos   = positions['bottom_side_pos']
        drawer_anchor_pos = positions['bottom_anchor_pos']
        tomato1_pos       = positions['tomato1']
        tomato2_pos       = positions['tomato2']
        plate_pos         = positions['plate']

        # === Task Plan Execution ===

        # 1) Rotate gripper to 90 degrees about its local Z axis
        print("[Task] Rotating gripper to 90 degrees...")
        # quaternion for 90° about Z: [x, y, z, w]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        
        # === Plan Steps (Frozen) ===
        obs, reward, done = rotate(env, task, target_quat)
        obs, reward, done = move(env, task, target_pos=drawer_side_pos)
        obs, reward, done = move(env, task, target_pos=drawer_anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        # === End Frozen Steps ===

        # 8) Place tomato2 onto plate
        print(f"[Task] Placing tomato2 onto plate at: {plate_pos}")
        obs, reward, done = place(env, task, target_pos=plate_pos)
        if done:
            print("[Task] Completed successfully! Reward:", reward)
        else:
            print("[Task] Task finished (done=False). Final reward:", reward)

    except Exception as e:
        print("[Task] Exception occurred during execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()