import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Move to side position of the bottom drawer
        side_bottom = positions['bottom_side_pos']
        print("[Task] Step 1: Moving to bottom side position:", side_bottom)

        # Step 2: Rotate gripper 90 degrees about the Z axis to align with drawer handle
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("[Task] Step 2: Rotating gripper to 90° about Z axis:", target_quat)

        [Frozen Code Start]
        obs, reward, done = move(env, task, target_pos=np.array(side_bottom)
obs, reward, done = rotate(env, task, target_quat=np.array(target_quat)
        [Frozen Code End]
        if done:
            print("[Task] Episode ended after step 2.")
            return

        # Step 3: Move to the anchor position of the bottom drawer
        anchor_bottom = positions['bottom_anchor_pos']
        print("[Task] Step 3: Moving to bottom anchor position:", anchor_bottom)
        obs, reward, done = move(env, task, target_pos=np.array(anchor_bottom))
        if done:
            print("[Task] Episode ended after step 3.")
            return

        # Step 4: Pull the bottom drawer open
        pull_distance = 0.05  # meters
        print("[Task] Step 4: Pulling drawer open by", pull_distance, "m")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Episode ended after step 4.")
            return

        # Steps 5–8: Pick two items (tomatoes) and place them onto the plate
        plate_pos = positions['plate']
        for idx, item in enumerate(['item1', 'item2'], start=5):
            obj_pos = positions[item]
            print(f"[Task] Step {idx}: Picking up {item} at {obj_pos}")
            obs, reward, done = pick(env, task, target_pos=np.array(obj_pos))
            if done:
                print(f"[Task] Episode ended after pick of {item}.")
                return
            print(f"[Task] Step {idx+1}: Placing {item} onto plate at {plate_pos}")
            obs, reward, done = place(env, task, target_pos=np.array(plate_pos))
            if done:
                print(f"[Task] Episode ended after place of {item}.")
                return

        print("[Task] Plan execution complete. Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()