import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        positions = get_object_positions()

        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper to ninety degrees
        try:
            print("[Plan] Step 1: rotate to ninety degrees")
            # create a target quaternion corresponding to 90° about Z
            target_quat = R.from_euler('xyz', [0, 0, 90], degrees=True).as_quat()
            obs, reward, done = rotate(env, task, target_quat)
        except Exception as e:
            print(f"[Error] rotate failed: {e}")
            raise

        if done:
            print("[Plan] Task ended after rotate.")
            return

        # Step 2: Move to side-pos-bottom
        try:
            print("[Plan] Step 2: move to side-pos-bottom")
            pos_side = positions.get('side-pos-bottom', positions.get('bottom_side_pos'))
            obs, reward, done = move(env, task, np.array(pos_side))
        except Exception as e:
            print(f"[Error] move to side-pos-bottom failed: {e}")
            raise

        if done:
            print("[Plan] Task ended after move to side-pos-bottom.")
            return

        # Step 3: Move to anchor-pos-bottom
        try:
            print("[Plan] Step 3: move to anchor-pos-bottom")
            pos_anchor = positions.get('anchor-pos-bottom', positions.get('bottom_anchor_pos'))
            obs, reward, done = move(env, task, np.array(pos_anchor))
        except Exception as e:
            print(f"[Error] move to anchor-pos-bottom failed: {e}")
            raise

        if done:
            print("[Plan] Task ended after move to anchor-pos-bottom.")
            return

        # Step 4: Pull drawer to open it
        try:
            print("[Plan] Step 4: pull drawer")
            obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        except Exception as e:
            print(f"[Error] pull failed: {e}")
            raise

        if done:
            print("[Plan] Task ended after pull.")
            return

        # Steps 5-10: Pick and place each tomato onto plate
        tomato_keys = ['tomato1', 'tomato2']
        plate_key = 'plate'
        for idx, t_key in enumerate(tomato_keys, start=1):
            # Move to tomato
            try:
                step_num = 5 + (idx - 1) * 3
                print(f"[Plan] Step {step_num}: move to {t_key}")
                pos_t = positions.get(t_key, positions.get(f'item{idx}'))
                obs, reward, done = move(env, task, np.array(pos_t))
            except Exception as e:
                print(f"[Error] move to {t_key} failed: {e}")
                raise
            if done:
                print(f"[Plan] Task ended after move to {t_key}.")
                return

            # Pick tomato
            try:
                step_num += 1
                print(f"[Plan] Step {step_num}: pick {t_key}")
                obs, reward, done = pick(env, task, np.array(pos_t))
            except Exception as e:
                print(f"[Error] pick {t_key} failed: {e}")
                raise
            if done:
                print(f"[Plan] Task ended after pick {t_key}.")
                return

            # Move to plate
            try:
                step_num += 1
                print(f"[Plan] Step {step_num}: move to {plate_key}")
                pos_plate = positions.get(plate_key)
                obs, reward, done = move(env, task, np.array(pos_plate))
            except Exception as e:
                print(f"[Error] move to {plate_key} failed: {e}")
                raise
            if done:
                print(f"[Plan] Task ended after move to {plate_key}.")
                return

            # Place tomato on plate
            try:
                step_num += 1
                print(f"[Plan] Step {step_num}: place {t_key} on {plate_key}")
                obs, reward, done = place(env, task, np.array(pos_plate))
            except Exception as e:
                print(f"[Error] place {t_key} failed: {e}")
                raise
            if done:
                print(f"[Plan] Task ended after place {t_key}.")
                return

        print("[Plan] All steps executed. Task completed successfully! Reward:", reward)

    except Exception as main_e:
        print("[Fatal] An exception occurred during plan execution:", main_e)
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()