[Frozen Code Start]
[Frozen Code End]

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    '''Combined-domain task: unlock bottom drawer and place tomatoes on the plate.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Expected keys: 'bottom_side_pos', 'bottom_anchor_pos', 'tomato1', 'tomato2', 'plate'

        # === Oracle Plan Execution ===

        # Step 1: Move to bottom side position
        print("[Plan] Step 1: Moving to bottom side position")
        side_pos = positions['bottom_side_pos']
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Plan] Early termination after move-to-side. Reward:", reward)
            return

        # Step 2: Rotate gripper to ninety_deg
        print("[Plan] Step 2: Rotating gripper by 90 degrees")
        # create quaternion representing 90-degree rotation around X axis
        target_quat = R.from_euler('x', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Early termination after rotate. Reward:", reward)
            return

        # Step 3: Move to bottom anchor position
        print("[Plan] Step 3: Moving to bottom anchor position")
        anchor_pos = positions['bottom_anchor_pos']
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Plan] Early termination after move-to-anchor. Reward:", reward)
            return

        # Step 4: Pull drawer to open it
        print("[Plan] Step 4: Pulling the bottom drawer to open it")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Plan] Early termination after pull. Reward:", reward)
            return

        # Step 5: Pick the first tomato
        print("[Plan] Step 5: Picking up first tomato (tomato1)")
        tomato1_pos = positions['tomato1']
        obs, reward, done = pick(env, task, tomato1_pos)
        if done:
            print("[Plan] Early termination after pick(tomato1). Reward:", reward)
            return

        # Step 6: Place the first tomato onto the plate
        print("[Plan] Step 6: Placing first tomato onto the plate")
        plate_pos = positions['plate']
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Early termination after place(tomato1). Reward:", reward)
            return

        # Step 7: Pick the second tomato
        print("[Plan] Step 7: Picking up second tomato (tomato2)")
        tomato2_pos = positions['tomato2']
        obs, reward, done = pick(env, task, tomato2_pos)
        if done:
            print("[Plan] Early termination after pick(tomato2). Reward:", reward)
            return

        # Step 8: Place the second tomato onto the plate
        print("[Plan] Step 8: Placing second tomato onto the plate")
        obs, reward, done = place(env, task, plate_pos)
        if done:
            print("[Plan] Early termination after place(tomato2). Reward:", reward)
            return

        print("[Plan] Completed all steps. Tomatoes are on the plate, other items remain undisturbed.")
        print("[Plan] Final reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()
