import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Drawer interaction positions
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']
        # Tomato and plate positions
        tomato1_pos = positions['tomato1']
        tomato2_pos = positions['tomato2']
        plate_pos = positions['plate']

        # 1) Determine target orientation for gripper rotation
        if 'ninety_deg' in positions:
            target_quat = positions['ninety_deg']
        else:
            raise KeyError("Required orientation 'ninety_deg' not found in object positions.")

        # 2) Estimate pull distance for opening the drawer
        if 'bottom_joint_pos' in positions:
            joint_pos = positions['bottom_joint_pos']
            pull_distance = np.linalg.norm(bottom_anchor_pos - joint_pos)
        else:
            pull_distance = 0.2  # fallback distance

        # ====================================================
        # === Execute the Oracle Plan Step by Step ===
        # Goal: Unlock the bottom drawer then move tomatoes to the plate
        # ====================================================

        [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=target_quat)
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        obs, reward, done = move(env, task, target_pos=bottom_anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        obs, reward, done = pick(env, task, target_pos=tomato1_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        obs, reward, done = pick(env, task, target_pos=tomato2_pos)
        obs, reward, done = place(env, task, target_pos=plate_pos)
        [Frozen Code End]

        # Check final status
        if done:
            print("[Task] Task ended early with reward:", reward)
        else:
            print("[Task] Completed oracle plan successfully. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()