import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Prepare Oracle Plan Parameters ===
        # 1) Orientation for opening drawer
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        # 2) Side and anchor positions for bottom drawer
        side_bottom    = np.array(positions['bottom_side_pos'])
        anchor_bottom  = np.array(positions['bottom_anchor_pos'])
        # 3) Pull parameters
        pull_distance = 0.1
        pull_axis     = 'x'
        # 4) Table object positions
        tomato1_pos = np.array(positions['tomato1'])
        tomato2_pos = np.array(positions['tomato2'])
        plate_pos   = np.array(positions['plate'])

        # === Execute Oracle Plan ===
        try:
            # === Frozen Code Start ===
            obs, reward, done = rotate(env, task, target_quat)
            obs, reward, done = move(env, task, side_bottom)
            obs, reward, done = move(env, task, anchor_bottom)
            obs, reward, done = pull(env, task, pull_distance, pull_axis)
            obs, reward, done = pick(env, task, tomato1_pos)
            obs, reward, done = place(env, task, plate_pos)
            obs, reward, done = pick(env, task, tomato2_pos)
            obs, reward, done = place(env, task, plate_pos)
            # === Frozen Code End ===

            # Check outcome after each block of actions
            if done:
                print("[Task] Plan terminated early. Reward:", reward)
                return

            # If we reach here, all steps executed without early termination
            print("[Task] Completed all plan steps successfully. Final reward:", reward)

        except Exception as e:
            print("[Task] Exception during plan execution:", str(e))

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()