import numpy as np
from scipy.spatial.transform import Rotation as R
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers (optional)
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve Object Positions
        positions = get_object_positions()
        # Choose a drawer to open (bottom)
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        # Fruits and plate
        tomato1_pos = np.array(positions['tomato1'])
        tomato2_pos = np.array(positions['tomato2'])
        plate_pos   = np.array(positions['plate'])

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper to 90 degrees around X to align with drawer
        print("[Plan] Step 1: rotate to 90 deg about X")
        ninety_quat = R.from_euler('x', 90, degrees=True).as_quat()
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("Early termination during rotate")
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Plan] Step 2: move to side of bottom drawer")
        obs, reward, done = move(env, task, target_pos=bottom_side_pos)
        if done:
            print("Early termination during move to side")
            return

        # Step 3: Pull the drawer open along its local -Y axis (pull distance ~0.2m)
        print("[Plan] Step 3: pull drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.2, pull_axis='-y')
        if done:
            print("Early termination during pull")
            return

        # Step 4: Pick up tomato1 from inside the opened drawer
        print("[Plan] Step 4: pick tomato1")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("Early termination during pick tomato1")
            return

        # Step 5: Place tomato1 on the plate
        print("[Plan] Step 5: place tomato1 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("Early termination during place tomato1")
            return
    [Frozen Code End]

        # Step 6: Pick up tomato2 from inside the opened drawer
        print("[Plan] Step 6: pick tomato2")
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.05,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("Early termination during pick tomato2")
            return

        # Step 7: Place tomato2 on the plate
        print("[Plan] Step 7: place tomato2 on plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.10,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("Early termination during place tomato2")
            return

        # Task complete
        print("===== Task completed: drawer opened and tomatoes placed on plate =====")
        print("Final reward:", reward)

    except Exception as e:
        print("Exception during task execution:", e)
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()