import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import pick, place, move

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers if needed
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        tomato1_pos = positions.get('tomato1')
        tomato2_pos = positions.get('tomato2')
        plate_pos = positions.get('plate')

        if tomato1_pos is None or tomato2_pos is None or plate_pos is None:
            raise ValueError("Could not find positions for tomato1, tomato2, or plate.")

        # === Execute Oracle Plan ===

        # Step 1: pick tomato1 from table
        print("[Plan] Step 1: pick tomato1 from table")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(tomato1_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early after picking tomato1.")
            return

        # Step 2: place tomato1 onto plate
        print("[Plan] Step 2: place tomato1 onto plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(plate_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early after placing tomato1.")
            return

        # Step 3: pick tomato2 from table
        print("[Plan] Step 3: pick tomato2 from table")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(tomato2_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early after picking tomato2.")
            return

        # Step 4: place tomato2 onto plate
        print("[Plan] Step 4: place tomato2 onto plate")
        obs, reward, done = place(
            env,
            task,
            target_pos=np.array(plate_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Plan executed, but 'done' flag is False. Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()