import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor
from env import setup_environment, shutdown_environment
from skill_code import pick, place, move, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # Ensure required objects exist
        if 'tomato1' not in positions or 'plate' not in positions:
            raise KeyError("Required objects 'tomato1' or 'plate' not found in positions.")
        tomato1_pos = np.array(positions['tomato1'], dtype=float)
        plate_pos = np.array(positions['plate'], dtype=float)

        # Step 1: Pick tomato1 from the table
        print("[Task] Step 1: Picking up tomato1 at:", tomato1_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato1_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely during pick of tomato1. Reward:", reward)
            return

        # Step 2: Place tomato1 on the plate
        print("[Task] Step 2: Placing tomato1 on plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely during place of tomato1. Reward:", reward)
            return

        # Step 3: Pick tomato2 from the table
        if 'tomato2' not in positions:
            raise KeyError("Required object 'tomato2' not found in positions.")
        tomato2_pos = np.array(positions['tomato2'], dtype=float)
        print("[Task] Step 3: Picking up tomato2 at:", tomato2_pos)
        obs, reward, done = pick(
            env,
            task,
            target_pos=tomato2_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task ended prematurely during pick of tomato2. Reward:", reward)
            return

        # Step 4: Place tomato2 on the plate
        print("[Task] Step 4: Placing tomato2 on plate at:", plate_pos)
        obs, reward, done = place(
            env,
            task,
            target_pos=plate_pos,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Task completed successfully! Final reward:", reward)
        else:
            print("[Task] Task not completed after placing tomato2. Final reward:", reward)

    except Exception as e:
        print("[Task] Exception during execution:", str(e))
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()