import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")

    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We expect at least these keys:
        #   'bottom_side_pos', 'bottom_anchor_pos'
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # === Plan-Specific Helpers ===
        # We need quaternions for 'zero_deg' and 'ninety_deg' rotations
        from scipy.spatial.transform import Rotation as R
        angle_quats = {
            'zero_deg': R.from_euler('z', 0, degrees=True).as_quat(),
            'ninety_deg': R.from_euler('z', 90, degrees=True).as_quat()
        }

        # === Execute Oracle Plan ===

        # Step 1: rotate(gripper, zero_deg, ninety_deg)
        print("[Task] Step 1: rotate to ninety degrees")
        obs, reward, done = rotate(env, task, angle_quats['ninety_deg'])
        if done:
            print("[Task] Ended early after rotate. Reward:", reward)
            return

        # Step 2: move(gripper, side-pos-bottom)
        print("[Task] Step 2: move to bottom side position:", bottom_side_pos)
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Ended early after move to side. Reward:", reward)
            return

        # Step 3: move(gripper, anchor-pos-bottom)
        print("[Task] Step 3: move to bottom anchor position:", bottom_anchor_pos)
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended early after move to anchor. Reward:", reward)
            return

        # Step 4: pick(gripper, anchor-pos-bottom)
        print("[Task] Step 4: pick at bottom anchor position")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Ended early after pick. Reward:", reward)
            return

        # Step 5: pull(gripper, bottom)
        # We choose a pull distance sufficient to open the drawer fully
        pull_distance = 0.15
        print("[Task] Step 5: pull drawer by distance:", pull_distance)
        obs, reward, done = pull(env, task, pull_distance)
        if done:
            print("[Task] Ended after pull. Reward:", reward)
        else:
            print("[Task] Pull completed; drawer should now be open. Reward:", reward)

    except Exception as e:
        print("Error during task execution:", str(e))
    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()