import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import move, pick, place, rotate, pull

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Task logic for selecting a drawer and opening it fully using predefined skills.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing the simulation (optional)
        init_video_writers(obs)

        # Wrap step and get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # Retrieve object positions
        positions = get_object_positions()
        # Select the bottom drawer's side and anchor positions
        side_pos = positions.get('bottom_side_pos')
        anchor_pos = positions.get('bottom_anchor_pos')
        if side_pos is None or anchor_pos is None:
            raise KeyError("Required positions for bottom drawer not found in object positions.")

        # 1) Rotate gripper from zero_deg to ninety_deg (90° around Z axis)
        # Quaternion for 90° around Z is [0, 0, sin(45°), cos(45°)]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Task] Rotating gripper to 90° orientation.")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Task ended during rotation.")
            return

        # 2) Move gripper to the side position of the bottom drawer
        print(f"[Task] Moving to side position: {side_pos}")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Task ended during move to side position.")
            return

        # 3) Move gripper to the anchor (handle) position of the bottom drawer
        print(f"[Task] Moving to anchor position: {anchor_pos}")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Task ended during move to anchor position.")
            return

    [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat)
obs, reward, done = move(env, task, side_pos)
obs, reward, done = move(env, task, anchor_pos)
    [Frozen Code End]

        # 4) Pick (grasp) the drawer handle
        print(f"[Task] Grasping drawer at: {anchor_pos}")
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        if done:
            print("[Task] Task ended during pick.")
            return

        # 5) Pull the drawer to open it fully
        # Compute pull distance along X using waypoint1 if available
        waypoint = positions.get('waypoint1')
        if waypoint is not None:
            pull_distance = abs(waypoint[0] - anchor_pos[0])
        else:
            pull_distance = 0.1  # fallback distance
        print(f"[Task] Pulling drawer by {pull_distance:.3f}m along X axis.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Task ended during pull.")
            return

        print("[Task] Drawer opened successfully!")

    except Exception as e:
        print(f"[Task] Exception occurred: {e}")
    finally:
        # Ensure environment teardown
        shutdown_environment(env)
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()