import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # Step 1: Rotate gripper to 90 degrees around Y axis
        from scipy.spatial.transform import Rotation as R
        target_quat_90 = R.from_euler('y', 90, degrees=True).as_quat()
        print("[Task] Step 1: Rotate gripper to 90 degrees")
        obs, reward, done = rotate(env, task, target_quat_90)
        if done:
            print("[Task] Early termination after rotate. Reward:", reward)
            return

        # Step 2: Move to bottom drawer side position
        bottom_side_pos = positions['bottom_side_pos']
        print(f"[Task] Step 2: Move to bottom side position: {bottom_side_pos}")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Task] Early termination after move to side. Reward:", reward)
            return

        # Step 3: Move to bottom drawer anchor position
        bottom_anchor_pos = positions['bottom_anchor_pos']
        print(f"[Task] Step 3: Move to bottom anchor position: {bottom_anchor_pos}")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Early termination after move to anchor. Reward:", reward)
            return

        # Step 4: Pick the drawer handle at anchor position
        print(f"[Task] Step 4: Pick at anchor position: {bottom_anchor_pos}")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return

        # Step 5: Pull to open the drawer
        bottom_joint_pos = positions['bottom_joint_pos']
        pull_distance = np.linalg.norm(bottom_joint_pos - bottom_anchor_pos)
        print(f"[Task] Step 5: Pull the drawer by distance: {pull_distance}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Drawer opened successfully. Reward:", reward)
        else:
            print("[Task] Drawer opened (done=False). Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()