from numpy import array
import numpy as np
from env import setup_environment, shutdown_environment
from skill_code import move, pick, place, rotate, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset the task to its initial state and set up recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()
        side_pos = np.array(positions['bottom_side_pos'])
        anchor_pos = np.array(positions['bottom_anchor_pos'])

        # Prepare for plan execution
        print("[Plan] Preparing to execute oracle plan")
        obs = task.get_observation()
        current_quat = obs.gripper_pose[3:7]

        # Execute the oracle plan steps
        # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat=current_quat)
        obs, reward, done = move(env, task, target_pos=side_pos)
        obs, reward, done = move(env, task, target_pos=anchor_pos)
        obs, reward, done = pick(env, task, target_pos=anchor_pos)
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        # Frozen Code End

        if done:
            print("[Plan] Execution terminated early. Reward:", reward)
            return
        print("[Plan] Oracle plan completed successfully. Final reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()