import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We choose the bottom drawer (unlocked) to open
        bottom_side_pos = positions['bottom_side_pos']
        bottom_anchor_pos = positions['bottom_anchor_pos']

        # Save initial observation quaternion for rotate step
        initial_obs = obs
        initial_quat = initial_obs.gripper_pose[3:7]

        # === Execute Oracle Plan ===

        # Step 1: Rotate gripper to side orientation
        # (Frozen Code Start)
        obs, reward, done = rotate(env, task, target_quat=initial_quat)
obs, reward, done = move(env, task, target_pos=np.array(bottom_side_pos)
        # (Frozen Code End)
        if done:
            print("[Task] Early termination after move-to-side.")
            return

        # Step 3: Move to anchor position of bottom drawer
        print(f"[Task] Step 3: Move to anchor position at {bottom_anchor_pos}")
        obs, reward, done = move(env, task, target_pos=np.array(bottom_anchor_pos))
        if done:
            print("[Task] Early termination after move-to-anchor.")
            return

        # Step 4: Pick the drawer handle at the anchor position
        print(f"[Task] Step 4: Pick drawer handle at {bottom_anchor_pos}")
        obs, reward, done = pick(
            env,
            task,
            target_pos=np.array(bottom_anchor_pos),
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after pick.")
            return

        # Step 5: Pull the drawer open
        pull_distance = 0.04  # approximate drawer depth
        print(f"[Task] Step 5: Pull drawer by {pull_distance}m along x-axis")
        obs, reward, done = pull(
            env,
            task,
            pull_distance=pull_distance,
            pull_axis='x',
            max_steps=100,
            threshold=0.01,
            timeout=10.0
        )
        if done:
            print("[Task] Early termination after pull.")
            return

        # Final status
        if not done:
            print("[Task] Drawer should now be open. Reward:", reward)
        else:
            print("[Task] Task ended with done=True. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()