import numpy as np
from scipy.spatial.transform import Rotation as R
from env import setup_environment, shutdown_environment
from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions

def run_skeleton_task():
    '''Execute the oracle plan: pull open any drawer that is not locked.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for recording (optional)
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve object positions
        positions = get_object_positions()

        # Step 1: rotate gripper to 90° about z-axis
        print("[Plan] Step 1: rotate gripper to 90° about z-axis")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()

        # Prepare bottom-drawer waypoints
        bottom_side = np.array(positions['bottom_side_pos'])
        bottom_anchor = np.array(positions['bottom_anchor_pos'])

    # Frozen Code Start
        obs, reward, done = rotate(env, task, target_quat)
obs, reward, done = move(env, task, bottom_side)
obs, reward, done = move(env, task, bottom_anchor)
    # Frozen Code End

        if done:
            print("[Plan] Terminated early during initial motions.")
            return

        # Step 4: pick bottom drawer anchor
        print("[Plan] Step 4: pick bottom drawer anchor")
        obs, reward, done = pick(env, task, bottom_anchor)
        if done:
            print("[Plan] Terminated early after pick on bottom anchor.")
            return

        # Step 5: pull bottom drawer open
        print("[Plan] Step 5: pull bottom drawer open")
        bottom_joint = np.array(positions['bottom_joint_pos'])
        vec = bottom_joint - bottom_anchor
        idx = int(np.argmax(np.abs(vec)))
        axis_map = {0: 'x', 1: 'y', 2: 'z'}
        axis = axis_map[idx]
        if vec[idx] < 0:
            axis = '-' + axis
        pull_distance = float(abs(vec[idx]))
        obs, reward, done = pull(env, task, pull_distance, pull_axis=axis)
        if done:
            print("[Plan] Terminated early after pulling bottom drawer.")
            return

        print("[Plan] Bottom drawer opened successfully. Reward:", reward)

    except Exception as e:
        print(f"[Error] Exception during plan execution: {e}")
    finally:
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()