import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset and begin recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)
        
        # Retrieve object positions
        positions = get_object_positions()
        
        # === Oracle Plan Execution ===
        
        # Step 1: Rotate gripper to ninety_deg
        print("[Task] Step 1: Rotating gripper to ninety_deg")
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, ninety_quat)
        if done:
            print("[Task] Early termination after rotate. Reward:", reward)
            return
        
        # Step 2: Move to bottom side position
        print("[Task] Step 2: Moving to bottom side position")
        side_pos = np.array(positions['bottom_side_pos'])
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Early termination after move to side_pos. Reward:", reward)
            return
        
        # Step 3: Move to bottom anchor position
        print("[Task] Step 3: Moving to bottom anchor position")
        anchor_pos = np.array(positions['bottom_anchor_pos'])
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Early termination after move to anchor_pos. Reward:", reward)
            return
        
        # Step 4: Pick bottom drawer handle
        print("[Task] Step 4: Picking bottom drawer handle")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Early termination after pick. Reward:", reward)
            return


        obs, reward, done = rotate(env, task, ninety_quat)
        obs, reward, done = move(env, task, side_pos)
        obs, reward, done = move(env, task, anchor_pos)
        obs, reward, done = pick(env, task, anchor_pos)


        # Step 5: Pull the drawer open
        print("[Task] Step 5: Pulling bottom drawer open")
        obs, reward, done = pull(env, task, pull_distance=0.1, pull_axis='x')
        if done:
            print("[Task] Task completed successfully! Reward:", reward)
        else:
            print("[Task] Task finished without 'done' flag. Reward:", reward)

    finally:
        shutdown_environment(env)
    
    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()