# run_skeleton_task.py (Fully Generic Skeleton)
#
# Description:
#  - This code is a skeleton that shows how to set up an environment, retrieve
#    object positions, and execute a plan without being tied to a specific skill
#    like pick or place.
#  - Use this as a base for creating your own task logic.
#  - The sections that are task-dependent (e.g., manipulation steps) are marked
#    with TODO or placeholders.

import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *

from video import init_video_writers, recording_step, recording_get_observation

# This module is assumed to provide object positions or other info about the environment
from object_positions import get_object_positions


def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()

        # === Oracle Plan Execution ===

        # Step 1: Rotate gripper from zero_deg to ninety_deg
        print("[Task] Step 1: Rotating gripper to ninety_deg orientation")
        # Approximate ninety_deg rotation about local z-axis as quaternion [0,0,sin(45°),cos(45°)]
        target_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Task] Finished during rotate step. Reward:", reward)
            return

        # Step 2: Move gripper to bottom side position
        print("[Task] Step 2: Moving gripper to bottom side position")
        side_pos = positions.get('bottom_side_pos')
        if side_pos is None:
            raise KeyError("Position 'bottom_side_pos' not found in object_positions")
        obs, reward, done = move(env, task, side_pos)
        if done:
            print("[Task] Finished during move-to-side step. Reward:", reward)
            return

        # Step 3: Move gripper to bottom anchor position
        print("[Task] Step 3: Moving gripper to bottom anchor position")
        anchor_pos = positions.get('bottom_anchor_pos')
        if anchor_pos is None:
            raise KeyError("Position 'bottom_anchor_pos' not found in object_positions")
        obs, reward, done = move(env, task, anchor_pos)
        if done:
            print("[Task] Finished during move-to-anchor step. Reward:", reward)
            return

        # Step 4: Pick (grasp) the bottom drawer at its anchor
        print("[Task] Step 4: Picking up the bottom drawer at anchor position")
        obs, reward, done = pick(env, task, anchor_pos)
        if done:
            print("[Task] Finished during pick step. Reward:", reward)
            return

        # Step 5: Pull the drawer open along x-axis by 0.1 meters
        print("[Task] Step 5: Pulling the drawer open")
        pull_distance = 0.1
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Task] Finished during pull step. Reward:", reward)
            return

        # Final check
        print("[Task] Plan execution completed. Final reward:", reward)

    except Exception as e:
        print(f"[Task] Exception during execution: {e}")
        raise

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()