import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Generic skeleton for running any task in your simulation.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # (Optional) Initialize video writers for capturing your simulation
        init_video_writers(obs)

        # Wrap the task steps for recording (if needed)
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        # We select the bottom drawer for this task
        side_pos_bottom = np.array(positions['bottom_side_pos'])
        anchor_pos_bottom = np.array(positions['bottom_anchor_pos'])

        # === Oracle Plan Execution ===
        # Step 1: Rotate gripper from zero_deg to ninety_deg
        # Represent zero_deg as identity quaternion and ninety_deg around Z-axis
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])
        print("[Task] Step 1: Rotating gripper to ninety_deg orientation.")
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
        if done:
            print("[Task] Finished early after rotation. Reward:", reward)
            return

        # Step 2: Move to the side position of the bottom drawer
        print(f"[Task] Step 2: Moving to side position of bottom drawer at {side_pos_bottom}.")
        obs, reward, done = move(env, task, target_pos=side_pos_bottom)
        if done:
            print("[Task] Finished early after move-to-side. Reward:", reward)
            return

        # Step 3: Move to the anchor position of the bottom drawer
        print(f"[Task] Step 3: Moving to anchor position of bottom drawer at {anchor_pos_bottom}.")
        obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        if done:
            print("[Task] Finished early after move-to-anchor. Reward:", reward)
            return

        # [Frozen Code Start]
    obs, reward, done = rotate(env, task, target_quat=ninety_quat)
obs, reward, done = move(env, task, target_pos=side_pos_bottom)
obs, reward, done = move(env, task, target_pos=anchor_pos_bottom)
        # [Frozen Code End]

        # Step 4: Pick the drawer handle at the anchor position
        print(f"[Task] Step 4: Picking the drawer handle at anchor position {anchor_pos_bottom}.")
        obs, reward, done = pick(
            env,
            task,
            target_pos=anchor_pos_bottom,
            approach_distance=0.15,
            max_steps=100,
            threshold=0.01,
            approach_axis='z',
            timeout=10.0
        )
        if done:
            print("[Task] Finished early after pick. Reward:", reward)
            return

        # Step 5: Pull to open the drawer fully
        pull_distance = 0.2
        print(f"[Task] Step 5: Pulling the drawer by {pull_distance} units along 'x' axis.")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis='x')
        if done:
            print("[Task] Finished early after pull. Reward:", reward)
            return

        # Final status check
        if obs is not None and getattr(obs, 'is_open_bottom', True):
            print("[Task] Drawer opened successfully! Reward:", reward)
        else:
            print("[Task] Drawer open status uncertain. Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")


if __name__ == "__main__":
    run_skeleton_task()