import numpy as np
from pyrep.objects.shape import Shape
from pyrep.objects.proximity_sensor import ProximitySensor

from env import setup_environment, shutdown_environment

from skill_code import *
from video import init_video_writers, recording_step, recording_get_observation

from object_positions import get_object_positions

def run_skeleton_task():
    '''Run the oracle plan to slide open the bottom drawer.'''
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        descriptions, obs = task.reset()
        init_video_writers(obs)
        # wrap step and get_observation for recording
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_bottom_pos = positions['bottom_side_pos']
        anchor_bottom_pos = positions['bottom_anchor_pos']
        joint_bottom_pos = positions['bottom_joint_pos']

        # === Define Quaternions for zero_deg and ninety_deg about Z-axis ===
        zero_quat = np.array([0.0, 0.0, 0.0, 1.0])
        ninety_quat = np.array([0.0, 0.0, np.sin(np.pi/4), np.cos(np.pi/4)])

        # === Execute Oracle Plan ===
        done = False

    [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=ninety_quat)
    obs, reward, done = move(env, task, target_pos=side_bottom_pos)
    [Frozen Code End]

        if done:
            print("[Plan] Episode ended after move to side position.")
            return

        # Step 3: move to anchor position of the bottom drawer
        print("[Plan Step 3] move gripper to anchor-bottom position:", anchor_bottom_pos)
        obs, reward, done = move(env, task, target_pos=anchor_bottom_pos)
        if done:
            print("[Plan] Episode ended after move to anchor position.")
            return

        # Step 4: pick the bottom drawer handle at the anchor position
        print("[Plan Step 4] pick at anchor-bottom position")
        obs, reward, done = pick(env, task, target_pos=anchor_bottom_pos)
        if done:
            print("[Plan] Episode ended after pick.")
            return

        # Step 5: pull the drawer open
        diff = joint_bottom_pos - anchor_bottom_pos
        pull_distance = np.linalg.norm(diff)
        axis_idx = int(np.argmax(np.abs(diff)))
        axis_map = {0: 'x', 1: 'y', 2: 'z'}
        sign = '' if diff[axis_idx] >= 0 else '-'
        pull_axis = sign + axis_map[axis_idx]
        print(f"[Plan Step 5] pull drawer by {pull_distance:.3f} along axis {pull_axis}")
        obs, reward, done = pull(env, task, pull_distance=pull_distance, pull_axis=pull_axis)
        if done:
            print("[Plan] Episode ended after pull.")
            return

        # Final Check
        if not done:
            print("[Plan] Completed all steps. Drawer should now be open. Reward:", reward)
        else:
            print("[Plan] Done. Episode signaled done=True.")

    except Exception as e:
        print(f"[Error] Exception during task execution: {e}")
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()