from env import setup_environment, shutdown_environment
from skill_code import rotate, move, pick, pull
from video import init_video_writers, recording_step, recording_get_observation
from object_positions import get_object_positions
import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    env, task = setup_environment()
    try:
        # Reset environment and start recording
        descriptions, obs = task.reset()
        init_video_writers(obs)
        task.step = recording_step(task.step)
        task.get_observation = recording_get_observation(task.get_observation)

        # Retrieve the relevant drawer positions
        positions = get_object_positions()
        bottom_side_pos = np.array(positions['bottom_side_pos'])
        bottom_anchor_pos = np.array(positions['bottom_anchor_pos'])
        bottom_joint_pos = np.array(positions['bottom_joint_pos'])

        # Step 1: Rotate gripper to ninety degrees around Z to align with drawer handle
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()
        print("[Plan] Step 1: rotate to ninety_deg")
        obs, reward, done = rotate(env, task, target_quat)
        if done:
            print("[Plan] Task ended after rotate. Reward:", reward)
            return

        # Step 2: Move gripper to the side position of the bottom drawer
        print("[Plan] Step 2: move to side position")
        obs, reward, done = move(env, task, bottom_side_pos)
        if done:
            print("[Plan] Task ended after move to side. Reward:", reward)
            return

        # Step 3: Move from side to anchor position (front of drawer)
        print("[Plan] Step 3: move to anchor position")
        obs, reward, done = move(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after move to anchor. Reward:", reward)
            return

        # Step 4: Pick (grasp) the drawer handle at the anchor position
        print("[Plan] Step 4: pick drawer handle")
        obs, reward, done = pick(env, task, bottom_anchor_pos)
        if done:
            print("[Plan] Task ended after pick. Reward:", reward)
            return

        # Step 5: Pull the drawer open along the X axis by the joint travel distance
        pull_distance = np.linalg.norm(bottom_joint_pos - bottom_anchor_pos)
        print(f"[Plan] Step 5: pull drawer by distance {pull_distance:.3f}")
        obs, reward, done = pull(env, task, pull_distance, pull_axis='x')
        if done:
            print("[Plan] Task ended after pull. Reward:", reward)
            return

        # Final success check
        if obs is not None and not done:
            print("[Plan] Completed all steps. Task achieved goal of opening drawer.")
        else:
            print("[Plan] Completed steps but task indicates done:", done, "reward:", reward)

    except Exception as e:
        print("[Error] Exception during task execution:", str(e))
    finally:
        shutdown_environment(env)
        print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()