import numpy as np
from scipy.spatial.transform import Rotation as R

def run_skeleton_task():
    print("===== Starting Skeleton Task =====")
    
    # === Environment Setup ===
    env, task = setup_environment()
    try:
        # Reset the task to its initial state
        descriptions, obs = task.reset()

        # Initialize video recording if needed
        init_video_writers(obs)

        # Wrap the task step / get_observation for recording
        original_step = task.step
        task.step = recording_step(original_step)
        original_get_obs = task.get_observation
        task.get_observation = recording_get_observation(original_get_obs)

        # === Retrieve Object Positions ===
        positions = get_object_positions()
        side_pos = positions['bottom_side_pos']
        anchor_pos = positions['bottom_anchor_pos']

        # === Oracle Plan Execution ===

        # Step 1: rotate gripper by 90 degrees around Z to face the drawer
        print("[Task] Step 1: Rotating gripper to ninety degrees around Z")
        target_quat = R.from_euler('z', 90, degrees=True).as_quat()  # xyzw

        # Step 2: move gripper to the side position of the bottom drawer
        print(f"[Task] Step 2: Moving to side position: {side_pos}")

        # Step 3: move gripper to the anchor position of the bottom drawer
        print(f"[Task] Step 3: Moving to anchor position: {anchor_pos}")

        # Step 4: pick the drawer handle at the anchor position
        print(f"[Task] Step 4: Picking drawer at anchor position: {anchor_pos}")

    [Frozen Code Start]
        obs, reward, done = rotate(env, task, target_quat=target_quat)
    obs, reward, done = move(env, task, target_pos=np.array(side_pos)
    obs, reward, done = move(env, task, target_pos=np.array(anchor_pos)
    obs, reward, done = pick(env, task, target_pos=np.array(anchor_pos)
    [Frozen Code End]

        if done:
            print("[Task] Terminated during pick. Reward:", reward)
            return

        # Step 5: pull the drawer open along its X axis
        OPEN_DISTANCE = 0.2
        print(f"[Task] Step 5: Pulling drawer by {OPEN_DISTANCE} meters along X")
        obs, reward, done = pull(env, task, pull_distance=OPEN_DISTANCE, pull_axis='x')
        if done:
            print("[Task] Terminated during pull. Reward:", reward)
            return

        # All steps executed successfully
        print("[Task] Successfully opened the drawer fully. Final Reward:", reward)

    finally:
        # Always ensure the environment is properly shutdown
        shutdown_environment(env)

    print("===== End of Skeleton Task =====")

if __name__ == "__main__":
    run_skeleton_task()